<?php

use App\Models\Configuration;
use Illuminate\Support\Facades\Cache;

if (!function_exists('config_value')) {
    /**
     * Get a configuration value from the database.
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    function config_value($key, $default = null)
    {
        $configs = app('app_config');
        return $configs[$key] ?? $default;
    }
}

if (!function_exists('load_configurations')) {
    /**
     * Load all configurations into cache.
     *
     * @return array
     */
    function load_configurations()
    {
        $configurations = Configuration::all();
        
        $configArray = [];
        foreach ($configurations as $config) {
            $configArray[$config->key] = $config->typed_value;
        }
        
        // Store in cache
        Cache::forever('app_configurations', $configArray);
        
        return $configArray;
    }
}

if (!function_exists('refresh_configurations')) {
    /**
     * Refresh the configuration cache.
     *
     * @return array
     */
    function refresh_configurations()
    {
        Cache::forget('app_configurations');
        return load_configurations();
    }
}

if (!function_exists('currency_symbol')) {
    /**
     * Get the currency symbol based on config value or passed currency code.
     *
     * @param string|null $currency
     * @return string|null
     */
    function currency_symbol(?string $currency = null): ?string
    {
        $currency = $currency ?: config_value('store_currency', 'GBP');

        $symbols = [
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
            'PLN' => 'zł',
            'CAD' => 'C$',
            'AUD' => 'A$',
            'JPY' => '¥',
            'CHF' => 'CHF',
            'SEK' => 'kr',
            'NOK' => 'kr',
            'DKK' => 'kr',
        ];

        return $symbols[strtoupper($currency)] ?? null;
    }
}
